/*
 * A shared routine to assign link verify responsibilities to
 * various hosts in the fabric
 */

#include <stdio.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_fma_flags.h"
#include "lf_xbar32.h"
#include "lf_process_fabric.h"

/*
 * local prototypes
 */
static struct lf_nic *lf_alv_closest_nic(struct lf_xbar *start_xp,
  int in_port, void *sequence, int *distp, int *loadp, int *portp);


void
lf_assign_link_verifies(
  struct lf_fabric *fp)
{
  struct lf_xbar *xp;
  struct lf_xbar *xp2;
  int port2;
  union lf_node *np;
  struct lf_nic *nicp;
  int nic_port;
  struct lf_verifier *vp;
  struct lf_verifier *vp2;
  void *sequence;		/* used as "seen" in BFS */
  int dist;
  int load;
  int p;
  int x;

  sequence = (void *)0;

  /*
   * Clear out xbar verifiers
   */
  for (x=0; x<fp->num_xbars; ++x) {
    xp = fp->xbars[x];
    for (p=0; p<xp->num_ports; ++p) {
      xp->verifiers[p].ver_nicp = NULL;
    }
  }

  /*
   * Find a verifier for each link
   */
  for (x=0; x<fp->num_xbars; ++x) {
    xp = fp->xbars[x];

    for (p=0; p<xp->num_ports; ++p) {

      /* Where are we connected */
      np = xp->topo_ports[p];

      /* skip if no conn, conn is not up, or conn is NIC */
      if (np == NULL
	  || xp->link_state[p] != LF_LINK_STATE_UP
	  || np->ln_type != LF_NODE_XBAR) {
	continue;
      }
      
      /*
       * find closest, least loaded NIC not through this port.
       * We may already have a verifier, found while inspecting the 
       * xbar on the other end of this link.  If ours is better than
       * his, replace it!
       */
      sequence = (void *)((char *)sequence + 1);
      nicp = lf_alv_closest_nic(xp, p, sequence, &dist, &load, &nic_port);

      /* If no NICs to be found, skip it */
      if (nicp == NULL) continue;

      xp2 = LF_XBAR(np);
      port2 = xp->topo_rports[p];
      vp = &xp->verifiers[p];
      vp2 = &xp2->verifiers[port2];

      /* If our distance is better or dist is same and load better, use this */
      if (vp->ver_nicp == NULL
	  || dist < FP_XBAR(xp)->min_dist[p]
	  || (dist == FP_XBAR(xp)->min_dist[p]
	      && load < FP_XBAR(xp)->best_load[p])) {


	/* If there was a previous verifier, decrement his load */
	if (vp2->ver_nicp != NULL) {
	  --FP_NIC(vp2->ver_nicp)->alv_load;
	}

	/* Fill in both ends of link */
	vp->ver_nicp = nicp;
	vp->ver_port = nic_port;
	FP_XBAR(xp)->min_dist[p] = dist;
	FP_XBAR(xp)->best_load[p] = load;

	/* beware loopback! */
	if (vp != vp2) {
	  vp2->ver_nicp = LF_VERIFY_OTHER_END;
	  vp2->ver_port = 0;
	  FP_XBAR(xp2)->min_dist[port2] = dist;
	  FP_XBAR(xp2)->best_load[port2] = load;
	}

	++FP_NIC(nicp)->alv_load;
      }
    }
  }

  return;
}


/*
 * Find the closest downhill NIC with the lowest load
 * "Downhill" means we never visit xbars with a higher clos level
 */
static struct lf_nic *
lf_alv_closest_nic(
  struct lf_xbar *start_xp,
  int in_port,
  void *sequence,
  int *distp,
  int *loadp,
  int *portp)
{
  struct lf_xbar *work_list;
  struct lf_xbar *new_list;
  struct lf_xbar *xp;
  struct lf_xbar *xp2;
  union lf_node *np2;
  struct lf_nic *nicp;
  struct lf_nic *best_nicp;
  int dist;
  int best_load;
  int best_port;
  int p;

  best_load = 999999;
  best_nicp = NULL;
  best_port = 0;
  dist = 0;

  FP_XBAR(start_xp)->bf_next = NULL;
  new_list = start_xp;
  work_list = NULL;
  FP_XBAR(start_xp)->bf_last_visitor = sequence;	/* mark this as seen */
  FP_XBAR(start_xp)->bf_in_port = in_port;

  while (new_list != NULL) {
    work_list = new_list;
    new_list = NULL;

    while (work_list != NULL) {
      xp = work_list;
      work_list = FP_XBAR(xp)->bf_next;

      for (p=0; p<xp->num_ports; ++p) {

	/* don't violate quadrant disable rules */
	if (lf_xbar_qd_violation(xp, FP_XBAR(xp)->bf_in_port, p)) continue;

	/* don't go out the port we came in */
	if (p == FP_XBAR(xp)->bf_in_port) continue;

	/* don't traverse links that are not up */
	if (xp->link_state[p] != LF_LINK_STATE_UP) continue;

	np2 = xp->topo_ports[p];

	/* skip if not connected */
	if (np2 == NULL) continue;

	/* If a NIC found, see if this is the best load */
	if (np2->ln_type == LF_NODE_NIC) {
	  nicp = LF_NIC(np2);

	  /* see if we have a new winner */
	  if ((nicp->host->fma_flags & FMA_FLAG_CAN_VERIFY)
	      && FP_NIC(nicp)->alv_load < best_load) {
	    best_load = FP_NIC(nicp)->alv_load;
	    best_nicp = nicp;
	    best_port = xp->topo_rports[p];
	  }

	} else if (np2->ln_type == LF_NODE_XBAR) {
	  xp2 = LF_XBAR(np2);

	  /* possibly skip for various reasons */
	  if (FP_XBAR(xp2)->bf_last_visitor == sequence	/* already seen */
	      || xp2->clos_level > xp->clos_level) {	/* uphill */
	    continue;
	  }

	  /* Add this xbar to the new work list for next time around */
	  FP_XBAR(xp2)->bf_next = new_list;
	  new_list = xp2;
	  FP_XBAR(xp2)->bf_last_visitor = sequence;	/* mark this as seen */
	  FP_XBAR(xp2)->bf_in_port = xp->topo_rports[p];
	}
      }
    }

    /* If a host found this time through, we're done */
    if (best_nicp != NULL) {
      *distp = dist;
      *loadp = best_load;
      *portp = best_port;
      return best_nicp;
    }

    ++dist;
  }

  return NULL;		/* no joy in Mudville */
}
